// ################################################################################
//
//  ICM-42607-P Test v0.0
//
//  Released:  27/12/2024
//
//  Author: TechKnowTone
//
// ################################################################################
/*
    TERMS OF USE: This software is furnished "as is", without technical support, and
    with no warranty, expressed or implied, as to its usefulness for any purpose. In
    no event shall the author or copyright holder be liable for any claim, damages,
    or other liability, whether in an action of contract, tort or otherwise, arising
    from, out of or in connection with the software or the use or other dealings in
    the software.

    Board: ESP32 Dev Module v3.1.0

    This code confirms the function of the ICM-42607-P by reading and printing the
    contents of its accelerometer and gyro registers. It does not rely on the use of
    the Invensense library, in order to provide direct access to the ICM chip.
*/
// Declare libraries
#include <Wire.h>             // I2C comms library

// Configuration
String Release = "ICM-42607-P QuickTest";
String Date = "27/12/2024";

// Define a task for core 0

// Define constants
#define I2Cdel 10                   // I2C bus timing delay for ICM-42607-P
#define ICM_address 0x68            // ICM-42607-P I2C address (0x68 or 0x69)

// Declare and initialise global variables
int16_t AccRawX;                    // 2-byte value read from the MPU
int16_t AccRawY;                    // 2-byte value read from the MPU
int16_t AccRawZ;                    // 2-byte value read from the MPU
int16_t GyrRawX;                    // 2-byte value read from the MPU
int16_t GyrRawY;                    // 2-byte value read from the MPU
int16_t GyrRawZ;                    // 2-byte value read from the MPU
byte I2C_Err;                       // flag returned from I2C comms
bool I2C_ICM;                       // true if ICM is visible on I2C bus
int t0,t1;                          // microsecond timers

// --------------------------------------------------------------------------------

void setup() {
  // Setup basic functions
  Serial.begin(115200);     //Start the serial port at 115200 baud
  Wire.begin();             //Start the I2C bus as master

  // Test for ICM-42607-P 3-axis motion sensor and initialise
  Wire.setClock(400000);    // set I2C 400 kHz Fast-mode, default = 100 kHz
  Wire.beginTransmission(ICM_address);
  I2C_Err = Wire.endTransmission();
  if (I2C_Err == 0) {
    Serial.println(F("Initialising ICM..."));
     I2C_ICM = true; ICM_42607_Initialise();
  } else {I2C_ICM = false; Serial.println(F("ICM-42607-P Not Found!"));}

}

// --------------------------------------------------------------------------------

void loop() {
  // runs repeatedly
  if (I2C_ICM) {
      //#############################################################################
      // Sensor readings
      //#############################################################################
      // As all register addresse are sequential, we read all 12 registers in one go
      t0 = micros();
      Wire.beginTransmission(ICM_address);                    // Start communication with the MPU
      Wire.write(0x0B);                                       // Start reading ACCEL_XOUT_H at register OB hex
      I2C_Err = Wire.endTransmission(false);                  // End the transmission
      delayMicroseconds(I2Cdel);                              // Allow MPU time to respond

      Wire.requestFrom(ICM_address, 12);                       // Request 6 bytes from the MPU
      AccRawX = Wire.read()<<8|Wire.read();                   // Combine the two X-axis bytes to make one integer
      AccRawY = Wire.read()<<8|Wire.read();                   // Combine the two Y-axis bytes to make one integer
      AccRawZ = Wire.read()<<8|Wire.read();                   // Combine the two Z-axis bytes to make one integer

      // delay(1);
      // Wire.beginTransmission(ICM_address);                    // Start communication with the gyro
      // Wire.write(0x11);                                       // Start reading GYRO_XOUT_H at register 11 hex
      // I2C_Err = Wire.endTransmission();                       // End the transmission
      // delayMicroseconds(I2Cdel);                              // Allow MPU time to respond

      // Wire.requestFrom(ICM_address, 6);                       // Request 6 bytes from the gyro
      GyrRawX = Wire.read()<<8|Wire.read();                   // Combine the two bytes to make one integer
      GyrRawY = Wire.read()<<8|Wire.read();                   // Combine the two bytes to make one integer
      GyrRawZ = Wire.read()<<8|Wire.read();                   // Combine the two bytes to make one integer

      t1 = micros() - t0;   // t1 = 434µs @ 400kHz I2C, and 1466 µs @ 100kHz I2C
      Serial.print  ("Ax:" + String(AccRawX) + "\tAy:" + String(AccRawY) + "\tAz:" + String(AccRawZ));
      Serial.print  ("\tGx:" + String(GyrRawX) + "\tGy:" + String(GyrRawY) + "\tGz:" + String(GyrRawZ));
      Serial.println("\ttime " + String(t1) + "µs");
      delay(99);
  }
}

// --------------------------------------------------------------------------------

void ICM_42607_Initialise() {
  // set up gyro hardware
  // By default the MPU-6050 sleeps. So we have to wake it up.
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search.
  Wire.write(0x1F);                     // We want to write to the PWR_MGMT_0 register (6B hex)
  Wire.write(0b00001111);               // Turn ON gyros and accelerometers in Low Noise mode
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(200);               // Allow ICM time to turn ON sensors

  //Set the full scale of the gyro to +/- 250 degrees per second
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search.
  Wire.write(0x20);                     // We want to write to the GYRO_CONFIG register (1B hex)
  Wire.write(0b01101000);               // Set the gyro bits as +/-250dps full scale, 200Hz ODR
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond

  //Set the full scale of the accelerometer to +/- 4g.
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search.
  Wire.write(0x21);                     // We want to write to the ACCEL_CONFIG register (1A hex)
  Wire.write(0b01001000);               // Set the accel bits as +/- 4g full scale range, 200 Hz ODR
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond

  //Set gyro filtering to improve the raw data.
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search
  Wire.write(0x23);                     // We want to write to the CONFIG register (1A hex)
  Wire.write(0b00000100);               // Set Digital Low Pass Filter to ~53Hz
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond

  //Set accelerometer filtering to improve the raw data.
  Wire.beginTransmission(ICM_address);  // Start communication with the address found during search
  Wire.write(0x24);                     // We want to write to the CONFIG register (1A hex)
  Wire.write(0b00000100);               // Set 2x averaging and Low Pass Filter to ~53Hz
  I2C_Err = Wire.endTransmission();     // End the transmission with the MPU.
  if (I2C_Err != 0) {return;}
  delayMicroseconds(I2Cdel);                // Allow MPU time to respond
}

// --------------------------------------------------------------------------------

